#region References

using System;
using System.Reflection;
using System.Windows.Forms;
using System.Text;
using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.BOL;
using gov.va.med.vbecs.GUI.controls;
using Action = System.Action;

#endregion

namespace gov.va.med.vbecs.GUI
{
	/// <summary>
	/// Class VbecsBaseForm
	/// </summary>
	public class VbecsBaseForm : Form
	{
		#region Form Controls

		private bool _persistWindowState = false;

		/// <summary>
		/// vbecsLogo1
		/// </summary>
		protected VbecsLogo vbecsLogo1;

		/// <summary>
		/// lblRequiredField
		/// </summary>
		protected Label lblRequiredField;

		/// <summary>
		/// hlpHelp
		/// </summary>
		protected HelpProvider hlpHelp;

		/// <summary>
		/// ttToolTip
		/// </summary>
		protected ToolTip ttToolTip;

		/// <summary>
		/// **** 508 Compliance ***
		/// </summary>
		protected gov.va.med.vbecs.GUI.controls.VbecsErrorProvider errError; 

		private System.ComponentModel.IContainer components;

		#endregion

		#region Variables

        /// <summary>
        /// CR3480 - adding to replace IndysDialogResult return code
        /// </summary>
        private bool _didLocksExpire = false;
 
		/// <summary>
		/// Variable determining whether the form has been updated or changed.
		/// </summary>
		private bool _Dirty;
	
		/// <summary>
		/// Variable to keep recursive focus from occurring when changing focus in a leave event; A value of true indicates
		/// that the form has already started processing the leave event and it shouldn't try to do it again
		/// </summary>
		private bool _recursionGuard = false;

		/// <summary>
		/// Current (logged on) user.
		/// </summary>
		private static VbecsUser _thisUser;		

		/// <summary>
		/// Set to close the form without calling closing event handlers.
		/// </summary>
		private bool _forcedClosureWithoutConfirmation;

		private const long WM_GETMINMAXINFO = 0x24;

		/// <summary>
		/// PointApi
		/// </summary>
		public struct PointApi
		{
			/// <summary>
			/// x
			/// </summary>
			public int x;

			/// <summary>
			/// y
			/// </summary>
			public int y;
		}

		/// <summary>
		/// MinMaxInfo
		/// </summary>
		public struct MinMaxInfo
		{
			/// <summary>
			/// PTReserved
			/// </summary>
			public PointApi PTReserved; 

			/// <summary>
			/// PTMaxSize
			/// </summary>
			public PointApi PTMaxSize; 

			/// <summary>
			/// PTMaxPosition
			/// </summary>
			public PointApi PTMaxPosition;

			/// <summary>
			/// PTMinTrackSize
			/// </summary>
			public PointApi PTMinTrackSize; 

			/// <summary>
			/// PTMaxTrackSize
			/// </summary>
			public PointApi PTMaxTrackSize;
		}

		TextBoxBase editMenuTextBox = null;

		/// <summary>
		/// mnuCutCopyPaste ContextMenu
		/// </summary>
		protected System.Windows.Forms.ContextMenu mnuCutCopyPaste;

		/// <summary>
		/// vbecsWindowState PersistWindowState
		/// </summary>
		protected controls.PersistWindowState vbecsWindowState;

		private System.Windows.Forms.MenuItem cmnuUndo;
		private System.Windows.Forms.MenuItem menuItem12;
		private System.Windows.Forms.MenuItem cmnuCut;
		private System.Windows.Forms.MenuItem cmnuCopy;
		private System.Windows.Forms.MenuItem cmnuPaste;
		private System.Windows.Forms.MenuItem cmnuDelete;
		private System.Windows.Forms.MenuItem menuItem17;
		private System.Windows.Forms.MenuItem cmnuSelectAll;

		/// <summary>
		/// hlphelpChm HelpProvider
		/// </summary>
		protected System.Windows.Forms.HelpProvider hlphelpChm;
		

		#endregion

		#region Events

		/// <summary>
		/// VbecsStatusEventHandler Delegate
		/// </summary>
		public delegate void VbecsStatusEventHandler(object sender, System.EventArgs e);

		/// <summary>
		/// **** 508 Compliance ***
		/// KeyDownSpeakMessageEventHandler Delegate
		/// </summary>
		public delegate void KeyDownSpeakMessageEventHandler(object sender, System.EventArgs e);

		/// <summary>
		/// OnVbecsStatusChange Event
		/// </summary>
		public event VbecsStatusEventHandler OnVbecsStatusChange;

		/// <summary>
		/// **** 508 Compliance ***
		/// KeyDownSpeakMessage Event
		/// </summary>
		public event KeyDownSpeakMessageEventHandler KeyDownSpeakMessage;
		
		/// <summary>
		/// Fired when an update or change has taken place on the form.
		/// </summary>
		protected void SetDirtyFlag(object sender, EventArgs e )
		{
			this.IsDirty = true;
			this.ResetLockTimer(sender, e);
		}

		/// <summary>
		/// Closing Event Handler
		/// </summary>
		/// <param name="e"></param>
		protected override void OnClosing(System.ComponentModel.CancelEventArgs e)
		{
			try
			{
                bool lockingFired = this._lockTimeOutFired || BOL.LockManager.HasBaseLockTimedout(this.BaseLockedFormId);

                // Determine if there have been changes made to the current base business object
                // Adding LockTimeOut check - we don't want to prompt for changes on a KABLAM
                if (this.IsDirty && !lockingFired && !_forcedClosureWithoutConfirmation)
                {
					// Alert the user if there have been changes asking to save the facility
					DialogResult _closingResult = GuiMessenger.ShowMessageBox( Common.StrRes.ConfMsg.Common.UnsavedData() );

					// If the user selects no, then cancel the event.
					if( _closingResult != DialogResult.Yes )
					{
						e.Cancel = true;
						return;
					}
				}

				DetachLocking();
			}
			catch (System.Data.SqlClient.SqlException exc)
			{
				FrmAppMainFormBase.ReportFatalApplicationException(exc,false);
			}

			ClearTooltipsProtectingFromMemoryLeakBug();

			if( !_forcedClosureWithoutConfirmation )
				base.OnClosing( e );
			else
				e.Cancel = true;
		}

		/// <summary>
		/// Mouse Down Event Handler
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void VbecsBaseForm_MouseDown(object sender, System.Windows.Forms.MouseEventArgs e)
		{
			this.ResetLockTimer(sender, e);
		}

		/// <summary>
		/// Key Press Event Handler
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void VbecsBaseForm_KeyPress(object sender, System.Windows.Forms.KeyPressEventArgs e)
		{
			this.ResetLockTimer(sender, e);

		}

		/// <summary>
		/// Keydown Event Handler
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void VbecsBaseForm_KeyDown(object sender, System.Windows.Forms.KeyEventArgs e)
		{
			if ( Common.LogonUser.IsVisuallyImpairedUser && e != null && e.KeyCode == Keys.F9 )
			{
				if (KeyDownSpeakMessage != null)
				{
					if(sender != null && e != null)
					{
						KeyDownSpeakMessage(sender, e);
					}
				}
			}
			//
			this.ResetLockTimer(sender, e);
		}

		/// <summary>
		/// Menu Cut Copy Paste Popup Event Handler 
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void mnuCutCopyPaste_Popup(object sender, System.EventArgs e)
		{
			try
			{
				editMenuTextBox = this.ActiveControl as TextBoxBase;

				if (editMenuTextBox != null)
				{
					IDataObject iData = Clipboard.GetDataObject();
				    if (iData != null) this.cmnuPaste.Enabled = iData.GetDataPresent(DataFormats.Text);
				    this.cmnuCut.Enabled =  (editMenuTextBox.SelectionLength > 0);
					this.cmnuCopy.Enabled =  (editMenuTextBox.SelectionLength > 0);
					this.cmnuDelete.Enabled =  (editMenuTextBox.SelectionLength > 0);
					this.cmnuUndo.Enabled = (editMenuTextBox.CanUndo);
					this.cmnuSelectAll.Enabled = (editMenuTextBox.Text.Length > 0);
				}
				else
				{
					this.cmnuCut.Enabled = false;
					this.cmnuCopy.Enabled = false;
					this.cmnuPaste.Enabled = false;
					this.cmnuDelete.Enabled = false;
					this.cmnuUndo.Enabled = false;
					this.cmnuSelectAll.Enabled = false;
				}
			}
			catch
			{
				editMenuTextBox = null;
				this.cmnuCut.Enabled = false;
				this.cmnuCopy.Enabled = false;
				this.cmnuPaste.Enabled = false;
				this.cmnuDelete.Enabled = false;
				this.cmnuUndo.Enabled = false;
				this.cmnuSelectAll.Enabled = false;
			}
		}

		/// <summary>
		/// Undo Click Event Handler
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void cmnuUndo_Click(object sender, System.EventArgs e)
		{
			if (editMenuTextBox != null)
			{
				if (editMenuTextBox.CanUndo)
				{
					editMenuTextBox.Undo();
					editMenuTextBox.ClearUndo();
				}
			}
		}

		/// <summary>
		/// Cut Click Event Handler
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void cmnuCut_Click(object sender, System.EventArgs e)
		{
			if (editMenuTextBox != null)
			{
				editMenuTextBox.Cut();
			}

		}

		/// <summary>
		/// Copy Click Event Handler
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void cmnuCopy_Click(object sender, System.EventArgs e)
		{
			if (editMenuTextBox != null)
			{
				editMenuTextBox.Copy();
			}

		}

		/// <summary>
		/// Paste Click Event Handler
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void cmnuPaste_Click(object sender, System.EventArgs e)
		{
			if (editMenuTextBox != null)
			{
				editMenuTextBox.Paste();
			}

		}

		/// <summary>
		/// Menu Delete Click Event Handler
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void cmnuDelete_Click(object sender, System.EventArgs e)
		{
			if(( editMenuTextBox != null ) && (!editMenuTextBox.ReadOnly))
			{
				editMenuTextBox.Text = editMenuTextBox.Text.Remove(editMenuTextBox.SelectionStart,editMenuTextBox.SelectionLength);
			}

		}

		/// <summary>
		/// Menu Select All Click Event Handler
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void cmnuSelectAll_Click(object sender, System.EventArgs e)
		{
			if (editMenuTextBox != null)
			{
				editMenuTextBox.SelectAll();
			}
		}

		/// <summary>
		/// Form Load Event Handler
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void VbecsBaseForm_Load(object sender, System.EventArgs e)
		{
			hlpHelp.HelpNamespace = Environment.CurrentDirectory + "\\VBECS.chm";
		}

		/// <summary>
		/// Mouse Move Event Handler
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void VbecsBaseForm_MouseMove(object sender, System.Windows.Forms.MouseEventArgs e)
		{
			this.ResetLockTimer(this, System.EventArgs.Empty);
		}

		/// <summary>
		/// Mouse Leave Event Handler
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void VbecsBaseForm_MouseLeave(object sender, System.EventArgs e)
		{
			this.ResetLockTimer(this, System.EventArgs.Empty);
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>06/29/2005</CreationDate>
		/// <summary>
		/// Used as a work-around to fix the memory leak 'feature'
		/// in the Date Picker Control.  Basically, the control
		/// needs to be told to dispose or it will cause the 
		/// parent form (and any preceding parents) to stay
		/// in memory.  
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		protected void VbecsBaseForm_Closed(object sender, EventArgs e)
		{
            if (_formIsNonModal)        //LOCKING 2.0 - I liked my code until I had to do this...bleh!
            {
                this.RefreshParentLockTimeEvent -= new EventHandler(this._parentForm.ResetLockTimer);

                //Did the lockout happen during our shift?
                if (!this._lockTimeOutFired && !BOL.LockManager.HasBaseLockTimedout(this.BaseLockedFormId))
                {
                    this._parentForm.UnPauseLockingTimer();
                }
            }

			for ( int idx = 0; idx < this.Controls.Count; idx++ )
			{
				CheckForAndDisposeDateControls( this.Controls[idx] );
			}
		}

		#endregion

		#region Constructors/Destructors

		/// <summary>
		/// Constructor
		/// </summary>
		public VbecsBaseForm() : this( true ) {}

		/// <summary>
		/// Constructor
		/// </summary>
		/// <param name="persistWindowState"></param>
		public VbecsBaseForm( bool persistWindowState )
		{			
			// Required for Windows Form Designer support
			InitializeComponent();

			Application.Idle += new System.EventHandler(OnIdle);

			hlpHelp.HelpNamespace = Environment.CurrentDirectory+"\\VBECS.chm";
			this.hlphelpChm.HelpNamespace = Environment.CurrentDirectory+"\\VBECS.chm";

			this._persistWindowState = persistWindowState;

			if( !persistWindowState || (this.GetType().Name == "FrmCollectionFacilityMaintain"))
				return;

			if( !persistWindowState )
				return;

			// Set RegistryPath for VBECS GUI form
			string registryPath = @"Software\GOV\VA\MED\GUI\" + this.GetType().Name;

			// Create PersistWindowState object to save and restore form location, size and window state to registry
			this.vbecsWindowState = new controls.PersistWindowState( this, registryPath );			
		}

		/// <summary>
		/// Clean up any resources being used.
		/// </summary>
		protected override void Dispose( bool disposing )
		{
			if( disposing )
			{
				if(components != null)
				{
					components.Dispose();
				}
			}
			base.Dispose( disposing );
		}


		#endregion

		#region Windows Form Designer generated code
		/// <summary>
		/// Required method for Designer support - do not modify
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{
			this.components = new System.ComponentModel.Container();
			System.Resources.ResourceManager resources = new System.Resources.ResourceManager(typeof(VbecsBaseForm));
			this.vbecsLogo1 = new gov.va.med.vbecs.GUI.VbecsLogo();
			this.lblRequiredField = new System.Windows.Forms.Label();
			this.hlpHelp = new System.Windows.Forms.HelpProvider();
			this.ttToolTip = new ToolTip(this.components);
			this.errError = new gov.va.med.vbecs.GUI.controls.VbecsErrorProvider(); // **** 508 Compliance ***
			this.mnuCutCopyPaste = new System.Windows.Forms.ContextMenu();
			this.cmnuUndo = new System.Windows.Forms.MenuItem();
			this.menuItem12 = new System.Windows.Forms.MenuItem();
			this.cmnuCut = new System.Windows.Forms.MenuItem();
			this.cmnuCopy = new System.Windows.Forms.MenuItem();
			this.cmnuPaste = new System.Windows.Forms.MenuItem();
			this.cmnuDelete = new System.Windows.Forms.MenuItem();
			this.menuItem17 = new System.Windows.Forms.MenuItem();
			this.cmnuSelectAll = new System.Windows.Forms.MenuItem();
			this.hlphelpChm = new System.Windows.Forms.HelpProvider();
			this.SuspendLayout();
			// 
			// vbecsLogo1
			// 
			this.vbecsLogo1.Dock = System.Windows.Forms.DockStyle.Top;
			this.vbecsLogo1.Font = new System.Drawing.Font("Arial", 10F, System.Drawing.FontStyle.Bold);
			this.vbecsLogo1.ForeColor = System.Drawing.Color.AntiqueWhite;
			this.vbecsLogo1.Location = new System.Drawing.Point(0, 0);
			this.vbecsLogo1.Name = "vbecsLogo1";
			this.vbecsLogo1.Size = new System.Drawing.Size(642, 30);
			this.vbecsLogo1.TabIndex = 0;
			this.vbecsLogo1.Title = null;
			// 
			// lblRequiredField
			// 
			this.lblRequiredField.AutoSize = true;
			this.lblRequiredField.Dock = System.Windows.Forms.DockStyle.Bottom;
			this.lblRequiredField.Location = new System.Drawing.Point(0, 457);
			this.lblRequiredField.Name = "lblRequiredField";
			this.lblRequiredField.Size = new System.Drawing.Size(85, 16);
			this.lblRequiredField.TabIndex = 2;
			this.lblRequiredField.Text = "* Required Field";
			// 
			// ttToolTip
			// 
			this.ttToolTip.AutoPopDelay = 10000;
			this.ttToolTip.InitialDelay = 500;
			this.ttToolTip.ReshowDelay = 100;
			this.ttToolTip.ShowAlways = true;
			// 
			// errError
			// 
			this.errError.ContainerControl = this;
			this.errError.Icon = ((System.Drawing.Icon)(resources.GetObject("errError.Icon")));
			// 
			// mnuCutCopyPaste
			// 
			this.mnuCutCopyPaste.MenuItems.AddRange(new System.Windows.Forms.MenuItem[] {
																							this.cmnuUndo,
																							this.menuItem12,
																							this.cmnuCut,
																							this.cmnuCopy,
																							this.cmnuPaste,
																							this.cmnuDelete,
																							this.menuItem17,
																							this.cmnuSelectAll});
			this.mnuCutCopyPaste.Popup += new System.EventHandler(this.mnuCutCopyPaste_Popup);
			// 
			// cmnuUndo
			// 
			this.cmnuUndo.Index = 0;
			this.cmnuUndo.Shortcut = System.Windows.Forms.Shortcut.CtrlU;
			this.cmnuUndo.Text = "&Undo";
			this.cmnuUndo.Click += new System.EventHandler(this.cmnuUndo_Click);
			// 
			// menuItem12
			// 
			this.menuItem12.Index = 1;
			this.menuItem12.Text = "-";
			// 
			// cmnuCut
			// 
			this.cmnuCut.Index = 2;
			this.cmnuCut.Shortcut = System.Windows.Forms.Shortcut.CtrlX;
			this.cmnuCut.Text = "Cu&t";
			this.cmnuCut.Click += new System.EventHandler(this.cmnuCut_Click);
			// 
			// cmnuCopy
			// 
			this.cmnuCopy.Index = 3;
			this.cmnuCopy.Shortcut = System.Windows.Forms.Shortcut.CtrlC;
			this.cmnuCopy.Text = "&Copy";
			this.cmnuCopy.Click += new System.EventHandler(this.cmnuCopy_Click);
			// 
			// cmnuPaste
			// 
			this.cmnuPaste.Index = 4;
			this.cmnuPaste.Shortcut = System.Windows.Forms.Shortcut.CtrlV;
			this.cmnuPaste.Text = "&Paste";
			this.cmnuPaste.Click += new System.EventHandler(this.cmnuPaste_Click);
			// 
			// cmnuDelete
			// 
			this.cmnuDelete.Index = 5;
			this.cmnuDelete.Shortcut = System.Windows.Forms.Shortcut.Del;
			this.cmnuDelete.Text = "&Delete";
			this.cmnuDelete.Click += new System.EventHandler(this.cmnuDelete_Click);
			// 
			// menuItem17
			// 
			this.menuItem17.Index = 6;
			this.menuItem17.Text = "-";
			// 
			// cmnuSelectAll
			// 
			this.cmnuSelectAll.Index = 7;
			this.cmnuSelectAll.Shortcut = System.Windows.Forms.Shortcut.CtrlA;
			this.cmnuSelectAll.Text = "Select A&ll";
			this.cmnuSelectAll.Click += new System.EventHandler(this.cmnuSelectAll_Click);
			// 
			// VbecsBaseForm
			// 
			this.AutoScaleBaseSize = new System.Drawing.Size(5, 13);
			this.ClientSize = new System.Drawing.Size(642, 473);
			this.Controls.Add(this.lblRequiredField);
			this.Controls.Add(this.vbecsLogo1);
			this.Font = new System.Drawing.Font("Arial", 8.25F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, ((System.Byte)(0)));
			this.hlphelpChm.SetHelpNavigator(this, System.Windows.Forms.HelpNavigator.Topic);
			this.Icon = ((System.Drawing.Icon)(resources.GetObject("$this.Icon")));
			this.KeyPreview = true;
			this.Name = "VbecsBaseForm";
			this.hlphelpChm.SetShowHelp(this, true);
			this.ShowInTaskbar = false;
			this.SizeGripStyle = System.Windows.Forms.SizeGripStyle.Hide;
			this.StartPosition = System.Windows.Forms.FormStartPosition.CenterScreen;
			this.Text = "VBECS";
			this.KeyDown += new System.Windows.Forms.KeyEventHandler(this.VbecsBaseForm_KeyDown);
			this.MouseDown += new System.Windows.Forms.MouseEventHandler(this.VbecsBaseForm_MouseDown);
			this.Click += new System.EventHandler(this.ResetLockTimer);
			this.KeyPress += new System.Windows.Forms.KeyPressEventHandler(this.VbecsBaseForm_KeyPress);
			this.Load += new System.EventHandler(this.VbecsBaseForm_Load);
			this.DoubleClick += new System.EventHandler(this.ResetLockTimer);
            this.Paint += new System.Windows.Forms.PaintEventHandler(this.VbecsBaseForm_Paint);
			this.MouseMove += new System.Windows.Forms.MouseEventHandler(this.VbecsBaseForm_MouseMove);
			this.MouseLeave += new System.EventHandler(this.VbecsBaseForm_MouseLeave);
			this.ResumeLayout(false);
			this.Closed += new EventHandler(VbecsBaseForm_Closed);

		}
		#endregion

		#region Methods

		/// <summary>
		/// SpeakListViewWithCheckBox
		/// </summary>
		/// <param name="listView"></param>
		public void SpeakListViewWithCheckBox(System.Windows.Forms.ListView listView)
		{
			if (listView.SelectedIndices.Count > 0 && Common.LogonUser.IsVisuallyImpairedUser)
			{
				using (new WaitCursor())
				{
					System.Text.StringBuilder buildUserList = new StringBuilder();
					ListViewItem lVI = ((System.Windows.Forms.ListView) listView).SelectedItems[0];
					string[] mySubItems = new string[lVI.SubItems.Count];
					int subCnt = lVI.SubItems.Count;
					int colCnt = listView.Columns.Count;
					int nLoop = 0;
					if(subCnt < colCnt)
					{
						nLoop = subCnt;
					}
					else
					{
						nLoop = colCnt;
					}
					
					for(int nCtr = 0; nCtr < nLoop; nCtr++)
					{
						if(listView.Columns[nCtr].Text.Trim() != string.Empty)
						{
							buildUserList.Append(listView.Columns[nCtr].Text + "coe lin ");
							buildUserList.Append("  ");
						}
						mySubItems[nCtr] = lVI.SubItems[nCtr].Text.Trim();
						if(lVI.SubItems[nCtr].Text.Trim().Length == 0)
						{
							buildUserList.Append("");
						}
						else
						{
							buildUserList.Append(mySubItems[nCtr]);
						}
						buildUserList.Append(" ");
					}
					
					string textToSpeak =buildUserList.ToString();
					
					this.SpeakMessage( textToSpeak );	
				}		
			}
		}

		/// <newpara> Function Name: Ready</newpara>
		/// <newpara> Developers: Cameron Taylor</newpara>
		/// <newpara> Site Name: Hines OIFO </newpara>
		/// <newpara> Date Created: 4/30/2002</newpara>
		/// <summary> Description: Set system ready icon in MDI parent form status bar.</summary>
		/// <returns>N/A</returns>		
		public void Ready()
		{
			if (OnVbecsStatusChange!=null) 
			{
				OnVbecsStatusChange(this, new VbecsStatusEventArgs(gov.va.med.vbecs.GUI.VbecsStatus.Ready));
			}
		}

		/// <newpara> Function Name: Stopped</newpara>
		/// <newpara> Developers: Cameron Taylor</newpara>
		/// <newpara> Site Name: Hines OIFO </newpara>
		/// <newpara> Date Created: 4/30/2002</newpara>
		/// <summary> Description: Set system stopped icon in MDI parent form status bar</summary>
		/// <returns>N/A</returns>		
		public void Stopped()
		{
			if (OnVbecsStatusChange!=null) 
			{
				Common.Utility.MessageBeep();
				OnVbecsStatusChange(this, new VbecsStatusEventArgs(gov.va.med.vbecs.GUI.VbecsStatus.Error));
			}
		}

		/// <newpara> Function Name: Working</newpara>
		/// <newpara> Developers: Cameron Taylor</newpara>
		/// <newpara> Site Name: Hines OIFO </newpara>
		/// <newpara> Date Created: 4/30/2002</newpara>
		/// <summary> Description: Set system working icon in MDI parent form status bar</summary>
		/// <returns>N/A</returns>	
		public void Working()
		{
			if (OnVbecsStatusChange!=null) OnVbecsStatusChange(this, new VbecsStatusEventArgs(gov.va.med.vbecs.GUI.VbecsStatus.Working));
		}

		/// <summary>
		/// This method can be used by derived forms to set tooltips for form-owned controls 
		/// utilizing autogenerated string resource items. 
		/// </summary>
		/// <param name="controlToSetTooltipFor">Control to set tooltip for.</param>
		/// <param name="resourceItem">String resource item to extract text from.</param>
		protected void SetThisToolTip( Control controlToSetTooltipFor, StrRes.StringResourceItem resourceItem )
		{
			if( controlToSetTooltipFor == null )
				throw( new ArgumentNullException( "controlToSetTooltipFor" ) );

			//CR 2837 - Wrapping SetToolTip() with Active flag setting to avoid tooltip pop-ups without mouse hovering
            //CR3414: accessing control from other thread
            if (InvokeRequired)
            {
                BeginInvoke(new Action(() =>
                {
                    ttToolTip.Active = false;
                    ttToolTip.SetToolTip(controlToSetTooltipFor, resourceItem.ResString);
                    ttToolTip.Active = true;
                }));
            }
            else
            {
                ttToolTip.Active = false;
                ttToolTip.SetToolTip(controlToSetTooltipFor, resourceItem.ResString);
                ttToolTip.Active = true;
            }
		}

		/// <summary>
		/// This method can be used by derived forms to set help file mappings for form-owned controls 
		/// utilizing autogenerated string resource items. 
		/// </summary>
		/// <param name="controlToSetHelpFor"></param>
		/// <param name="resourceItem"></param>
		protected void SetThisHelpFile( Control controlToSetHelpFor, StrRes.StringResourceItem resourceItem)
		{
			if( controlToSetHelpFor == null )
			{
				throw( new ArgumentNullException( "controlToSetHelpFor" ) );
			}

			this.hlphelpChm.SetHelpKeyword(controlToSetHelpFor, resourceItem.ResString );
			this.hlphelpChm.SetHelpNavigator(controlToSetHelpFor, HelpNavigator.Topic);
			this.hlphelpChm.SetShowHelp(controlToSetHelpFor, true);
		}

		/// <summary>
		/// ForceToCloseWithoutConfirmation
		/// </summary>
		public void ForceToCloseWithoutConfirmation()
		{
			_forcedClosureWithoutConfirmation = true;

			Close();
		}
		
		/// <summary>
		/// This method is put in as workaround for memory leak issue suggested by MS consultant.
		/// </summary>
		private void ClearTooltipsProtectingFromMemoryLeakBug()
		{			
			ttToolTip.RemoveAll();
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>06/29/2005</CreationDate>
		/// <summary>
		/// Used as a work-around to fix the memory leak 'feature'
		/// in the Date Picker Control.
		/// </summary>
		/// <param name="control"></param>
		private void CheckForAndDisposeDateControls( Control control )
		{
			if( control != null )
			{
				if( control is DateTimePicker )
				{
					control.Parent = null;
					control.Dispose();
				}
				//
				if( this.HasChildren )
				{
					// Recursively call this method for each child control
					foreach( Control childControl in control.Controls )
					{
						CheckForAndDisposeDateControls( childControl );
					}
				}
			}
		}

		/// <summary>
		/// JV - Workaround for MdiChild bug
		/// </summary>
		/// <param name="m"></param>
		protected override void WndProc(ref System.Windows.Forms.Message m )
		{
				if(this.IsMdiChild && m.Msg == WM_GETMINMAXINFO)
				{
					MinMaxInfo mmi = (MinMaxInfo)m.GetLParam(typeof(MinMaxInfo));

					mmi.PTMinTrackSize.x = this.MinimumSize.Width;
					mmi.PTMinTrackSize.y = this.MinimumSize.Height;

					if(this.MaximumSize.Width != 0 || this.MaximumSize.Height != 0)
					{
						mmi.PTMaxTrackSize.x = this.MaximumSize.Width;
						mmi.PTMaxTrackSize.y = this.MaximumSize.Height;
					}

					System.Runtime.InteropServices.Marshal.StructureToPtr( mmi, m.LParam, true );
				}

				base.WndProc( ref m );
		}

		/// <summary>
		/// **** 508 Compliance ***
		/// </summary>
		/// <param name="message"></param>
		public void SpeakMessage(string message)
		{
			// Also get error provider messages, if any
			if ( this.errError != null )
			{
				string errorMessages = this.errError.GetErrorMessages();
				//
				if ( errorMessages != null && errorMessages.Length > 0 )
				{
					message = string.Concat(message, errorMessages);
				}
			}
			//
			if ( message != null && message.Length > 0 )
			{
				Common.Utility.Speak( message );
			}
		}

		#endregion

		#region Properties

		/// <summary>
		/// RequiredFieldsLabel
		/// </summary>
		public bool RequiredFieldsLabel
		{
			get
			{
				return lblRequiredField.Visible;
			}
			set
			{
				lblRequiredField.Visible = value;
			}
		}


		/// <summary>
		/// Sets currently logged on user.
		/// </summary>
		protected static VbecsUser SetThisUser 
		{
			set
			{
				_thisUser = value;
			}
		}

		/// <summary>
		/// Currently logged on user identity.
		/// </summary>
		public static VbecsUser ThisUser
		{
			get 
			{
				return _thisUser;
			}
		}

		/// <summary>
		/// Current dirty status of the form.
		/// </summary>
		public virtual bool IsDirty
		{
			get
			{ 
				return _Dirty; 
			}
			set 
			{ 
				this._Dirty = value;
			}
		}

		/// <summary>
		/// Current recursion status of the form
		/// </summary>
		public virtual bool RecursionGuard
		{
			get
			{
				return _recursionGuard;
			}
			set
			{
				this._recursionGuard = value;
			}
		}

		/// <summary>
		/// Checks whether currently logged on user is authorized for a given function.
		/// </summary>
		/// <param name="functionCode">Function code to check.</param>
		/// <returns>True if user is authorized for a given function code. Otherwise - false.</returns>
		public static bool IsCurrentUserAuthorizedForFunction( Common.FunctionCode functionCode )
		{
			if( _thisUser == null ) 
				return false;

			return _thisUser.IsAuthorizedForFunction( functionCode );
		}

		/// <summary>
		/// Checks whether currently logged on user is authorized for 
		/// any of function codes from a given list. 
		/// </summary>
		/// <param name="functionCodes">List of function codes to check.</param>
		/// <returns>True if a logged in user is authorized for any of given function codes.</returns>
		public static bool IsCurrentUserAuthorizedForAnyOfFunctions( params Common.FunctionCode[] functionCodes )
		{
			foreach( Common.FunctionCode _functionCode in functionCodes )
				if( IsCurrentUserAuthorizedForFunction( _functionCode ) )
					return true;

			return false;
		}

		/// <summary>
		/// Checks whether currently logged on user is authorized for 
		/// all function codes in a given list. 
		/// </summary>
		/// <param name="functionCodes">List of function codes to check.</param>
		/// <returns>True if a logged in user is authorized for all given function codes.</returns>
		public static bool IsCurrentUserAuthorizedForAllFunctions( params Common.FunctionCode[] functionCodes )
		{
			foreach( Common.FunctionCode _functionCode in functionCodes )
				if( !IsCurrentUserAuthorizedForFunction( _functionCode ) )
					return false;

			return true;
		}


		#endregion

		#region Locking



		#region Locking Variables

        ///// <summary>
        ///// Some UCs can be openeed multiple times.  We need a way to distinguish which
        ///// locks to release on the LockTable if we perform a UnlockAllRecordsForUseCaseUser  TODO: KILL THIS
        ///// </summary>
        //private System.Guid _formInstanceGuid = System.Guid.Empty;

		/// <summary>
		/// Some forms don't have Locks, but they still inherit from VBECSBaseForm, so
		/// we need this flag to distinguish the 2, and only display the Timeout message
		/// for the necessary forms.
		/// </summary>
		private bool _showLockTimeout = false;

		private static bool _displayedLockWarning = false;
		private static int _lockWarningFormId = 0;        //LOCKING2.0 - renamed

		/// <summary>
		/// How many minutes of inactivity before a lock times out?
		/// </summary>
		private static int _lockInactivityTimeout;

		/// <summary>
		/// How long have we been "idle"
		/// </summary>
		protected System.DateTime _idleTime = System.DateTime.MinValue;

		/// <summary>
		/// Need this to prevent the Cancel button's -- "You have unsaved data" messasge from popping
		/// </summary>
		private bool _lockTimeOutFired = false;


		/// <summary>
		/// Need a local instance of lock manager to keep track of the created locks for this form
		/// </summary>
		private BOL.LockManager _lockController = null;

        //LOCKING2.0
        private static int _lockedFormIdCounter = 0;      //Static variable to assign each form a unique number 
        private int _lockedFormId = -1;                   //Current form's ID
        private int _lockingBaseFormId = -1;        //If current form is part of a chain, this is the formID of the base(root) 
        private static bool _ignoreLockingPaintRequests = false;    //Allows paint events to be ignored for lock chains. Paint closes windows on lock timeout
                                                                    //When forms are repainted, it can do it in any order (e.g bottom/parent first)
        private bool _pauseLockingTimer = false;    //Varible to control the OnIdle timer counting.. Child forms tell the parent to pause to avoid multiple alerts
        private VbecsBaseForm _parentForm = null;   //Microsoft recommended keeping any external references at the module level.. not sure why I'm listening to them
        private bool _formIsNonModal = false;       //needed for hacky .ShowFromLockedParent()

		#endregion

		#region Locking-Events

		/// <summary>
		///	Reset locktimer from parent.
		/// </summary>
		public event EventHandler RefreshParentLockTimeEvent;

        /// <summary>
        /// LOCKING2.0 - Pausing the lock timer allows any child windows to timeout without creating a double-whammy of lock timeout
        /// messages. 
        /// </summary>
        protected void PauseLockingTimer()
        {
            this._pauseLockingTimer = true;
        }

        /// <summary>
        /// Allows the child to communicate with a parent - CR3480
        /// </summary>
        protected void ChildLockExpired()
        {
            this._didLocksExpire = true;
        }

        /// <summary>
        /// LOCKING2.0 - Method to stop the pause
        /// </summary>
        protected void UnPauseLockingTimer()
        {
            this.ResetLockTimer(this, System.EventArgs.Empty);  //Since action is taken, reset
            this._pauseLockingTimer = false;
        }

        /// <summary>
        /// LOCKING2.0 - Custom method needed to pause the parent lock timer and set the locking variables
        /// This custom ShowDialog also allows me tell if a Lock was expired
        /// </summary>
        /// <param name="parentForm"></param>
        /// <param name="willChildFormHaveLocks"></param>
        /// <returns></returns>
        public DialogResult ShowDialogFromLockedParent(Form parentForm, bool willChildFormHaveLocks)
        {
            DialogResult returnCode = DialogResult.Abort;

            this._parentForm = (VbecsBaseForm)parentForm;
            this.BaseLockedFormId = this._parentForm.BaseLockedFormId;
            this.LockController.BaseLockedFormId = this.BaseLockedFormId;
            BOL.LockManager.AddToLockAlertList(this.BaseLockedFormId, this.LockedFormId);

            this.RefreshParentLockTimeEvent += new EventHandler(this._parentForm.ResetLockTimer);

            //This form has no locks, but it's in control of the parent's locks
            if (willChildFormHaveLocks) this._parentForm.PauseLockingTimer();
            returnCode = this.ShowDialog(this._parentForm);
            this.RefreshParentLockTimeEvent -= new EventHandler(this._parentForm.ResetLockTimer);

            //Did the lockout happen during our shift?
            if (this._lockTimeOutFired || BOL.LockManager.HasBaseLockTimedout(this.BaseLockedFormId))
            {
                this._parentForm.ChildLockExpired();        //CR3480 - bubble up to parent
                this._didLocksExpire = true;                //CR3480 - set local value
                return (DialogResult.Cancel);
            }
            else
            {
                this._parentForm.UnPauseLockingTimer();
                return(returnCode);
            }
        }

        /// <summary>
        /// LOCKING2.0 - Same as ShowDialogFromLockedParent - lame, yes, BUT.. VBECS does have like 4 places where Show() is used.. YARGH!
        /// Maybe someday those can be reworked?
        /// </summary>
        /// <param name="parentForm"></param>
        /// <param name="willChildFormHaveLocks"></param>
        /// <returns></returns>
        public void ShowFromLockedParent(Form parentForm, bool willChildFormHaveLocks)
        {
            this._formIsNonModal = true;
            this._parentForm = (VbecsBaseForm)parentForm;
            this.BaseLockedFormId = this._parentForm.BaseLockedFormId;
            this.LockController.BaseLockedFormId = this.BaseLockedFormId;
            BOL.LockManager.AddToLockAlertList(this.BaseLockedFormId, this.LockedFormId);

            this.RefreshParentLockTimeEvent += new EventHandler(this._parentForm.ResetLockTimer);

            //This form has no locks, but it's in control of the parent's locks
            if (willChildFormHaveLocks) this._parentForm.PauseLockingTimer();
            this.Show();

            //Rest of the code is moved to the _Closed() event
        }

        /// <summary>
        /// CR3480 - Allows inheriting forms to check if locks expired during processing
        /// </summary>
        public bool DidLocksExpire
        {
            get
            {
                return (this._didLocksExpire);
            }
        }

		/// <summary>
		/// Reset lock timer from parent
		/// </summary>
		protected void OnRefreshParentLockTimeEvent()
		{
			if (RefreshParentLockTimeEvent != null)
				RefreshParentLockTimeEvent(this, System.EventArgs.Empty);
		}

		/// <summary>
		/// Fires on Application.Idle -- keeps track of each form's idle time
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		public void OnIdle(object sender, EventArgs e)
		{
			//_showTimeOut - only set on forms where Locks are CREATED
			if (!this._showLockTimeout) return;

            //Get out if the timer is paused
            if (this._pauseLockingTimer) return;

            //MinValue is the default value. ResetLockTimer() will set the initial value (Now) to _idleTime 
            if (this._idleTime != System.DateTime.MinValue)
            {
                System.TimeSpan idleSpan = System.DateTime.Now - this._idleTime;

#if DEBUG       //Using this to trap where user activity isn't being trapped for locking "idle-time"
                this.Text = idleSpan.ToString();
#endif
                //If the DlgLockWarning window is not already visible... 
                if (!_displayedLockWarning)
                {
                    //If there's less than a minute left on the lock, get ready to display the warning (which counts down from 59secs)
                    if ((LockInactivityTimeout - idleSpan.Minutes) <= 1)
                    {
                        //If there's less than 10 seconds left - expire the lock - don't waste time with a warning - will this ever happen? TODO
                        if ((60 - idleSpan.Seconds) < 10)
                        {
                            this.ResetLockTimer(this, System.EventArgs.Empty);
                            this.LockExpired();
                            return;
                        }

                        //Display the LockWarning
                        _displayedLockWarning = true;
                        GUI.controls.DlgLockWarning warnUser = new GUI.controls.DlgLockWarning(60 - idleSpan.Seconds, this.LockController.LockFunction);
                        warnUser.RequiredFieldsLabel = false;

                        //Save the guid of the form that owns the Warning
                        _lockWarningFormId = this.LockedFormId;
                        DialogResult keepAlive = warnUser.ShowDialog(this);
						warnUser.Dispose();
						_displayedLockWarning = false;	//for one dialog window

						this.ResetLockTimer(this, System.EventArgs.Empty);

						if (keepAlive == DialogResult.Cancel)
						{
							this.LockExpired();
						}
						//return;   //LOCKING2.0 - removed, not needed, code execution is done
					}
				}
				else
				{
                    //The current form is busy displaying the warning message, so we don't need to do anything
                    if (this.LockedFormId == _lockWarningFormId) return;

                    //The idle time is more then allowed, kill the lock (this could happen on parent forms where the child expired first, or DST)
                    if (idleSpan.Minutes >= LockInactivityTimeout)
					{
						this.ResetLockTimer(this, System.EventArgs.Empty);
						this.LockExpired();
					}
				}
			}
		}

		/// <summary>
		/// This event's made public so it can be reset from calling forms
		/// We use System.DateTime.Now over GetDivsionCurrentDateTime because
		/// we don't care about time zones -- just minutes/seconds. TODO: Care about daylight savings
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		public void ResetLockTimer(object sender, System.EventArgs e)
		{
			//Raise an event so children's activities can refresh the parents activity
			this.OnRefreshParentLockTimeEvent();
		
			if (!this._showLockTimeout) return;

			this._idleTime = System.DateTime.Now;
		}
		#endregion


		#region Locking-Methods


		private void DetachLocking()
		{
            BOL.LockManager.RemoveFromLockAlertList(this.LockedFormId);     //LOCKING2.0 - replaced form queue popping

			// Unhook this event, otherwise it keeps running until Dispose
			Application.Idle -= new System.EventHandler( OnIdle );
			
			this.UnlockAllRecords();
			this._lockController = null;
		}

        /// <summary>
        /// LOCKING2.0 - Locks an entire UC - uses the value set in LockController.LockFucntion
        /// </summary>
        /// <returns></returns>
        protected bool LockUseCase()
        {
            //If the _lockTimeoutFired = true, then we shouldn't be doing any more locking (the current form lock timer is done)
            //I've seen this problem in UC_20.  If the user is inside the updVials control, and a lock timeout occurs, the 
            //Leave Event of the updVials control will fire after after the locktimeout window is shut down which will create new locks TODO??
            if (this._lockTimeOutFired) return (true);

            if (this.LockController.LockFunction == Common.LockFunctions.NA)
            {
                throw new Exception(Common.StrRes.SysErrMsg.Lock.LockFunctionNotSet().ResString);
            }

            try
            {
                BOL.Lock tmpLock;

                tmpLock = this.LockController.LockUseCase(System.Guid.Empty, false);

                //There was a conflict
                if (tmpLock.LockConflictIndicator(this.LockedFormId))
                {
                    this.ShowLockConflictMessages(tmpLock);
                }
            }
            catch (BOL.LockRecordException err)
            {
                GuiMessenger.ShowMessageBox(Common.StrRes.InfoMsg.Common.Dynamic(err.Message));

                this.LockController.UnlockRecords();

                return false;
            }

            this._showLockTimeout = true;
            this.ResetLockTimer(this, System.EventArgs.Empty);

            return true;
        }

        /// <summary>
        /// LOCKING2.0 - Locks an entire UC for a guid (eg UC38 is locked for a patientguid)
        /// </summary>
        /// <param name="lockedRecordGuid"></param>
        /// <returns></returns>
        protected bool LockUseCase(System.Guid lockedRecordGuid)
        {
            //If the _lockTimeoutFired = true, then we shouldn't be doing any more locking (the current form lock timer is done)
            //I've seen this problem in UC_20.  If the user is inside the updVials control, and a lock timeout occurs, the 
            //Leave Event of the updVials control will fire after after the locktimeout window is shut down which will create new locks TODO??
            if (this._lockTimeOutFired) return (true);

            if (this.LockController.LockFunction == Common.LockFunctions.NA)
            {
                throw new Exception(Common.StrRes.SysErrMsg.Lock.LockFunctionNotSet().ResString);
            }

            try
            {
                BOL.Lock tmpLock;

                tmpLock = this.LockController.LockUseCase(lockedRecordGuid, false);

                //There was a conflict
                if (tmpLock.LockConflictIndicator(this.LockedFormId))
                {
                    this.ShowLockConflictMessages(tmpLock);
                }
            }
            catch (BOL.LockRecordException err)
            {
                GuiMessenger.ShowMessageBox(Common.StrRes.InfoMsg.Common.Dynamic(err.Message));

                this.LockController.UnlockRecords();

                return false;
            }

            this._showLockTimeout = true;
            this.ResetLockTimer(this, System.EventArgs.Empty);

            return true;
        }

        /// <summary>
        /// LOCKING2.0 - Locks the records attached to the LockController.RecordGuids collection
        /// </summary>
        /// <returns></returns>
        protected bool SetLocks()
        {
            //Just in case we're not passing in any locks, return true to avoid tripping up fail logic
            if (this.LockController.RecordGuids.Count == 0) return (true);

            //If the _lockTimeoutFired = true, then we shouldn't be doing any more locking (the current form lock timer is done)
            //I've seen this problem in UC_20.  If the user is inside the updVials control, and a lock timeout occurs, the 
            //Leave Event of the updVials control will fire after after the locktimeout window is shut down which will create new locks TODO??
            if (this._lockTimeOutFired) return (true);

            //System.Guid[] uc38Guid = { System.Guid.NewGuid() };
            //bool uc38Conflict = false;

            if (this.LockController.LockFunction == Common.LockFunctions.NA)
            {
                throw new Exception(Common.StrRes.SysErrMsg.Lock.LockFunctionNotSet().ResString);
            }

            try
            {
                //We don't use a giant transaction for record locks, so we can display more descriptive
                //messages to the user on lock failure
                foreach (System.Guid recordGuid in this.LockController.RecordGuids)
                {
                    this.LockRecord(recordGuid, false);
                }
            }
            catch (BOL.LockRecordException err)
            {
                GuiMessenger.ShowMessageBox(Common.StrRes.InfoMsg.Common.Dynamic(err.Message));

                this.LockController.UnlockRecords();

                return false;
            }

            this._showLockTimeout = true;
            this.ResetLockTimer(this, System.EventArgs.Empty);

            return true;
        }

		/// <summary>
		/// I have a separate method here for code reuse.  That's about it.
		/// </summary>
		/// <param name="recordGuids"></param>
		/// <returns></returns>
        protected bool SetLocks(params System.Guid[] recordGuids)
        {
            //In case we're not passing in any locks, don't set anything
            if (recordGuids.Length == 0) return (true);

            //If the _lockTimeoutFired = true, then we shouldn't be doing any more locking
            //I've seen this problem in UC_20.  If the user is inside the updVials control, and a lock timeout occurs, the 
            //Leave Event of the updVials control will fire after after the locktimeout window is shut down which will create new locks
            if (this._lockTimeOutFired) return (true);

            if (this.LockController.LockFunction == Common.LockFunctions.NA)
            {
                throw new Exception(Common.StrRes.SysErrMsg.Lock.LockFunctionNotSet().ResString);
            }

            //bool isUC38 = (this.LockController.LockFunction == Common.LockFunctions.UC038DefinePatientTransfusionRequirements);
            try
            {
                //We don't use a giant transaction for record locks, so we can display more descriptive
                //messages to the user on lock failure
                foreach (System.Guid recordGuid in recordGuids)
                {
                    this.LockRecord(recordGuid, false);
                }
            }
            catch (BOL.LockRecordException err)
            {
                GuiMessenger.ShowMessageBox(Common.StrRes.InfoMsg.Common.Dynamic(err.Message));

                //Since we're in here it means that we collided with an existing lock.
                //Rather than figure out which locks we did get (if any) before the collision, just unlock the everything we tried
                this.LockController.UnlockRecords(recordGuids, this.LockedFormId, this.LockController.LockFunction);

                return false;
            }

            this._showLockTimeout = true;
            this.ResetLockTimer(this, System.EventArgs.Empty);

            return true;
        }


		/// <summary>
		/// This method should only be called by the MUCs, which are division unspecific, and therefore
		/// pass-in a TRUE value to the LockRecord method of the LockManager class 
		/// </summary>
		/// <param name="recordGuids"></param>
		/// <returns></returns>
		protected bool SetMUCLocks(params System.Guid[] recordGuids)
		{
            //In case we're not passing in any locks, don't set anything
            if (recordGuids.Length == 0) return (true);

            if (this.LockController.LockFunction == Common.LockFunctions.NA)
            {
                throw new Exception(Common.StrRes.SysErrMsg.Lock.LockFunctionNotSet().ResString);
            }

			try
			{
				foreach (System.Guid recordGuid in recordGuids)
				{
					this.LockRecord(recordGuid, true);
				}
			}
			catch(BOL.LockRecordException err)
			{
				GuiMessenger.ShowMessageBox(Common.StrRes.InfoMsg.Common.Dynamic(err.Message));
				//We can't unlock any records because we don't have a division
				return false;
			}

			this._showLockTimeout = true;
			this.ResetLockTimer(this, System.EventArgs.Empty);

			return true;
		}

		/// <summary>
		/// Unlocks all records for the current use case and resets the Timeout indicator flag
		/// </summary>
		protected void UnlockAllRecords()
		{
			// Save a DB call - NA means it was never set, so no locks could have fired, so BAIL!
			if (this.LockController.LockFunction == Common.LockFunctions.NA) return;

			this._showLockTimeout = false;

            this.LockController.UnlockAllRecordsForUseCaseUser(this.LockController.LockFunction, this.LockedFormId, this.BaseLockedFormId);       //LOCKING2.0 - Added BaseLockedFormId
		}

		/// <summary>
		/// ReleaseLocks
		/// </summary>
		/// <param name="recordGuids"></param>
		protected void ReleaseLocks(params System.Guid[] recordGuids)
		{
			this.LockController.UnlockRecords(recordGuids, this.LockedFormId, this.LockController.LockFunction);

			//If they're manually removing all the locks, and there's none left, TURN OFF the notification flag
			if (this.LockController.LockCount == 0)
				this._showLockTimeout = false;

		}

        /// <summary>
        /// Partner of SetLocks()
        /// </summary>
        protected void ReleaseLocks()
        {
            this.LockController.UnlockRecords();       //Locking2.0 - before it used to have this.FormInstanceGuid

            //If they're manually removing all the locks, and there's none left, TURN OFF the notification flag
            if (this.LockController.LockCount == 0)
                this._showLockTimeout = false;
        }

		/// <summary>
		/// LOCKING2.0 - Tweaked a bit to use new properties/methods, removed this.Hide(), this.Close(), this.Dispose() - they caused .NET upgrade failure
		/// </summary>
        private void LockExpired()
		{
            BOL.LockManager.ExpiredBaseLockedFormIds.Add(this.BaseLockedFormId);

            this._lockTimeOutFired = true;

            this.UnlockAllRecords();

            //We are going to use our own dialog box here, to allow us to show it on the taskbar
            //This is because the timeout message sometimes appears under the main form, (if they minimize the
            //remoted desktop session) and the users won't know what happened. 
            controls.DlgLockTimeout dlg = new controls.DlgLockTimeout(Common.StrRes.InfoMsg.Common.LockTimeout(), this.LockController.LockFunction);

            _ignoreLockingPaintRequests = true;

            dlg.ShowDialog(this);

            if (dlg != null)
                dlg.Close();

            _ignoreLockingPaintRequests = false;
            this.Invalidate();
		}

	    /// <summary>
		/// Lock the requesting record -- must have other "inputs" supplied
		/// 
		/// </summary>
		/// <param name="recordGuid"></param>
		/// <param name="multiDivisionLock">MultiDivision = MUC03</param>
		/// <returns></returns>
        private void LockRecord(System.Guid recordGuid, bool multiDivisionLock)
        {
            //This will cover any BOL constructors that might set recordGuids to NewGuid()
            if (recordGuid == System.Guid.Empty) return;

            BOL.Lock tmpLock;

            //If code locked it once, don't lock it again
            if (this.LockController.IsRecordAlreadyLockedInUC(recordGuid, out tmpLock))
                return;

            tmpLock = this.LockController.LockRecord(recordGuid, multiDivisionLock);     //LOCKING2.0 - LockManager contains FormInstanceGuid now, removed as param

            //There was a conflict
            if (tmpLock.LockConflictIndicator(this.LockedFormId))
            {
                this.ShowLockConflictMessages(tmpLock);
            }
            //else -- Nothing - this means the lock was successfully created!   
        }

		#endregion

        /// <summary>
        /// LOCKING2.0- Framework fix? Using the Paint method instead the old (Locking1.0) Activate method (removed) to check for lock expiration
        /// If a top most form timesout, this will always get called - the Activate() method needed the user to click on the form
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void VbecsBaseForm_Paint(object sender, PaintEventArgs e)
        {
            //This routine is probably going to annoy developers debugging code.. Sorry guys! #RELEASE?
            //Ignore the paint logic when we're about to close a form (see LockExpired()) - sort like a cheap _isLoading variable
            if (_ignoreLockingPaintRequests)
            {
                System.Diagnostics.Debug.WriteLine(this.Name + "-Paint()-" + "ignored");
                return;
            }

            //Does this form have locks? If not, do nothing
            if (this.LockController.IsLockCarrier(this.LockedFormId))
            {
                //Did our lock chain expire? If not, do nothing
                if (BOL.LockManager.HasBaseLockTimedout(this.BaseLockedFormId))
                {
                    //Is this the lock chain originator? 
                    if (this.LockedFormId == this.BaseLockedFormId)
                    {
                        //Are all the child forms closed? If not, do nothing. 
                        //Trying to close forms in a queue-pop-on, pop-off fasion to avoid memory conflicts
                        if (BOL.LockManager.AllLockAlertFormsAreClosed(this.BaseLockedFormId))
                            this.Close();
                    }
                    else    //We're not the chain originator, but we're in the chain and expired
                        this.Close();
                }
            }
        }

		/// <summary>
		/// Throws LockRecordExceptions (if exist) which will be caught and display a messagebox to the user
		/// </summary>
		/// <param name="tmpLock">Lock object to process</param>
		private void ShowLockConflictMessages(BOL.Lock tmpLock)
		{
			//Have to do ToUpper Cuz DB and Windows might not be EXACTLY da same
			if (tmpLock.UserNTLoginId.ToUpper() != Common.LogonUser.LogonUserName.ToUpper())
			{
				throw new BOL.LockRecordException(Common.StrRes.SysErrMsg.Lock.RecordAlreadyLocked(tmpLock.UserName, Common.VBECSDateTime.FormatDateTimeString(tmpLock.LockCreatedTime)).ResString);
			}

			// If the impotent fields are all the same, let it slide
			if (tmpLock.LockFunctionId == this.LockController.LockFunction && this.LockedFormId == tmpLock.LockedFormId)
			{
                if (tmpLock.LockFunctionId == Common.LockFunctions.UC038DefinePatientTransfusionRequirements)
				{
                    //LOHSE HAVING FUN
					//Everything looks the same, it must be a same-user conflict for UC_38, so display to the user
					throw new BOL.LockRecordException(Common.StrRes.SysErrMsg.Lock.RecordLockedInDifferentArea(Common.VBECSDateTime.FormatDateTimeString(tmpLock.LockCreatedTime)).ResString);
                    //throw new BOL.LockRecordException(Common.StrRes.SysErrMsg.Lock.RecordLockedInDifferentArea2(BOL.LockManager.GUIFriendlyLockLocation(this.LockController.LockFunction), Common.VBECSDateTime.FormatDateTimeString(tmpLock.LockCreatedTime)).ResString);

				}
				else
				{
					//For all NON UC_38 locks, if the session is different, it's a same record lock
					if (tmpLock.SessionGuid != Common.LogonUser.VbecsSessionGuid)
					{
                        //LOHSE HAVING FUN
						throw new BOL.LockRecordException(Common.StrRes.SysErrMsg.Lock.RecordLockedInDifferentArea(Common.VBECSDateTime.FormatDateTimeString(tmpLock.LockCreatedTime)).ResString);
                        //throw new BOL.LockRecordException(Common.StrRes.SysErrMsg.Lock.RecordLockedInDifferentArea2(BOL.LockManager.GUIFriendlyLockLocation(this.LockController.LockFunction), Common.VBECSDateTime.FormatDateTimeString(tmpLock.LockCreatedTime)).ResString);
					}
				}
			}
			else
			{
                //LOHSE HAVING FUN
				throw new BOL.LockRecordException(Common.StrRes.SysErrMsg.Lock.RecordLockedInDifferentArea(Common.VBECSDateTime.FormatDateTimeString(tmpLock.LockCreatedTime)).ResString);
                //throw new BOL.LockRecordException(Common.StrRes.SysErrMsg.Lock.RecordLockedInDifferentArea2(BOL.LockManager.GUIFriendlyLockLocation(this.LockController.LockFunction), Common.VBECSDateTime.FormatDateTimeString(tmpLock.LockCreatedTime)).ResString);
			}
		}

		#region Locking-Properties


        /// <summary>
        /// LOCKING2.0 - ID of the form - simple counter - getting away from Guids
        /// </summary>
        private int LockedFormId
        {
            get
            {
                if (this._lockedFormId == -1)
                {
                    _lockedFormIdCounter = _lockedFormIdCounter + 1;
                    this._lockedFormId = _lockedFormIdCounter;
                }

                return (this._lockedFormId);
            }
        }

		/// <summary>
		/// Need this to save DB calls and check for locks WITHIN code. LOCKING2.0- made public for locking property shifting
		/// </summary>
		public BOL.LockManager LockController
		{
			get
			{
				if (this._lockController == null)
				{
                    //CR3311 lockInactivityTimeout added
                    this._lockController = new BOL.LockManager(this.LockedFormId, this.BaseLockedFormId, LockInactivityTimeout);  //LOCKING2.0 - new ctor()
				}
				return(this._lockController);
			}
		}

        /// <summary>
        /// LOCKING2.0 - Keeps the FormID of the base(parent) in a lock chain
        /// </summary>
        public int BaseLockedFormId
        {
            get
            {
                if (this._lockingBaseFormId == -1)
                    return (this.LockedFormId);
                else
                    return (this._lockingBaseFormId);
            }
            set
            {
                this._lockingBaseFormId = value;
            }
        }

		

		/// <summary>
		/// Don't SET ME! EVER! Unless you're DivisionChange() on the MainForm!!!!
		/// </summary>
		public static int LockInactivityTimeout
		{
			get
			{
				//default the value to 15 -- For the MUCs, which aren't division specific
				if (_lockInactivityTimeout == 0)
					_lockInactivityTimeout = BOL.LockManager.DefaultLockInactivityTimeoutMin;        //CR3311

				return(_lockInactivityTimeout);
			}
			set
			{
				_lockInactivityTimeout = value;
			}
		}



		#endregion


		#endregion
	}

	/// <summary>
	/// VbecsStatus
	/// </summary>
	public enum VbecsStatus 
	{ 
		/// <summary>
		/// Ready
		/// </summary>
		Ready, 
		/// <summary>
		/// Working
		/// </summary>
		Working, 
		/// <summary>
		/// Error
		/// </summary>
		Error 
	};

	/// <summary>
	/// Class VbecsStatusEventArgs
	/// </summary>
	public class VbecsStatusEventArgs : EventArgs
	{
		private VbecsStatus vbecsStatus;

		/// <summary>
		/// Constructor
		/// </summary>
		/// <param name="vbecsStatus"></param>
		public VbecsStatusEventArgs(VbecsStatus vbecsStatus):base()
		{
			this.vbecsStatus = vbecsStatus;
		}

		/// <summary>
		/// Get Status
		/// </summary>
		public VbecsStatus Status
		{
			get { return vbecsStatus; }
		}
	}
}
